<?php
require_once __DIR__ . '/../../../init.php';
require_once __DIR__ . '/../../../includes/gatewayfunctions.php';
require_once __DIR__ . '/../../../includes/invoicefunctions.php';

use WHMCS\Config\Setting;
use WHMCS\Database\Capsule;

$gatewayModuleName = "fastpaymentbd";
$gatewayParams = getGatewayVariables($gatewayModuleName);

if (!$gatewayParams['type']) {
    die("Module Not Activated");
}

// fastpaymentbd callback data
$invoiceId      = $_REQUEST['invoice'] ?? 0;
$transactionId  = $_REQUEST['transactionId'] ?? '';
$paymentAmount  = floatval($_REQUEST['paymentAmount'] ?? 0);
$paymentFee     = floatval($_REQUEST['paymentFee'] ?? 0);

// API Key from gateway config
$apikey = $gatewayParams['apiKey'];

// Log request
logTransaction($gatewayParams['FriendlyName'], $_REQUEST, "Callback Received");

// Verify with fastpaymentbd API
$verifyData = json_encode(["transaction_id" => $transactionId]);
$verifyUrl = 'https://pay.fastpaymentbd.com/api/payment/verify';

$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => $verifyUrl,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => $verifyData,
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        'API-KEY: ' . $apikey
    ],
]);
$response = curl_exec($ch);
$httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curlError  = curl_error($ch);
curl_close($ch);

$res = json_decode($response, true);

if ($res && isset($res['status']) && $res['status'] === "COMPLETED") {

    // Check Invoice exists
    $invoice = Capsule::table('tblinvoices')->where('id', $invoiceId)->first();
    if (!$invoice) {
        logTransaction($gatewayParams['FriendlyName'], $_REQUEST, "Invoice Not Found");
        die("Invoice Not Found");
    }

    // Currency conversion if needed
    $currencyRate = floatval($gatewayParams['currency_rate'] ?? 1);
    $clientCurrencyId = Capsule::table('tblclients')->where('id', $invoice->userid)->value('currency');
    $invoiceCurrencyCode = Capsule::table('tblcurrencies')->where('id', $clientCurrencyId)->value('code');

    if (strtoupper($invoiceCurrencyCode) !== 'BDT' && $currencyRate > 0) {
        // Convert BDT to invoice currency
        $paymentAmount = $paymentAmount / $currencyRate;
    }

    // Prevent duplicate transaction
    checkCbTransID($transactionId);

    // Make sure payment amount covers remaining balance
    $finalAmount = max($paymentAmount, $invoice->balance);

    // Add payment & mark invoice as Paid
    addInvoicePayment(
        $invoiceId,
        $transactionId,
        $finalAmount,
        $paymentFee,
        $gatewayModuleName
    );

    // Force run provisioning module if invoice is now Paid
    $invoiceStatus = Capsule::table('tblinvoices')->where('id', $invoiceId)->value('status');
    if ($invoiceStatus === 'Paid') {
        $orderIds = Capsule::table('tblorders')->where('invoiceid', $invoiceId)->pluck('id');
        foreach ($orderIds as $orderId) {
            // Accept Order (this runs module create/upgrade commands)
            localAPI('AcceptOrder', ['orderid' => $orderId, 'autosetup' => true]);
        }
    }

    logTransaction($gatewayParams['FriendlyName'], $_REQUEST, "Successful");

    // Redirect to invoice page
    $systemUrl = Setting::getValue('SystemURL');
    header("Location: " . $systemUrl . '/viewinvoice.php?id=' . $invoiceId);
    exit();

} else {
    $errorMessage = "Payment Verification Failed: ";
    if ($curlError) {
        $errorMessage .= "cURL Error: " . $curlError;
    } elseif ($httpStatus != 200) {
        $errorMessage .= "HTTP Status: " . $httpStatus . " Response: " . $response;
    } elseif (!empty($res['message'])) {
        $errorMessage .= "API Message: " . $res['message'];
    } else {
        $errorMessage .= "Unknown error. Raw: " . $response;
    }

    logTransaction($gatewayParams['FriendlyName'], $_REQUEST, "Failed - " . $errorMessage);
    die("Payment Failed: " . $errorMessage);
}
